## -*-Tcl-*-
 # ###################################################################
 #  HTML mode - tools for editing HTML documents
 # 
 #  FILE: "htmlEngine.tcl"
 #                                    created: 96-04-29 21.31.28 
 #                                last update: 99-04-24 13.18.40 
 #  Author: Johan Linde
 #  E-mail: <jlinde@telia.com>
 #     www: <http://www.theophys.kth.se/~jl/Alpha.html>
 #  
 # Version: 2.1.4
 # 
 # Copyright 1996-1999 by Johan Linde
 #  
 # This software may be used freely, and distributed freely, as long as the 
 # receiver is not obligated in any way by receiving it.
 #  
 # If you make improvements to this file, please share them!
 # 
 # ###################################################################
 ##

proc htmlEngine.tcl {} {}

proc htmlIsInteger {str} {
	return [regexp {^-?[0-9]+$} [string trim $str]]
}

# Checks to see if the current window is empty, except for whitespace.
proc htmlIsEmptyFile {} {
	return [catch {search -s -f 1 -r 1 {[^ \t\r\n]+} 0}]
}

# Removes all tags from a string.
proc htmlTagStrip {str} {
	regsub -all {<[^<>]*>} $str "" str
	return $str
}

# Quoting of strings for meta tags.
proc htmlQuote {str} {
	regsub -all "#" $str {#;} str
	regsub -all "\"" $str {#qt;} str
	regsub -all "<" $str {#lt;} str
	regsub -all ">" $str {#gt;} str
	return $str
}

proc htmlUnQuote {str} {
	regsub -all {#qt;} $str "\"" str
	regsub -all {#lt;} $str "<" str
	regsub -all {#gt;} $str ">" str
	regsub -all {#;} $str "#" str
	return $str
}


# Find the version number of a program.
# Returns 0 if any problem.
proc htmlGetVersion {sig} {
	if {![app::isRunning $sig] && [catch {app::launchBack $sig}]} {
		return 0
	}
	set vers [AEBuild -r '$sig' core getd ---- "obj{want:type('prop'),from:null(),form:'prop',seld:type('vers')}"]
# 	set vers [objectProperty 'MACS' vers "obj {want:type(file), seld:$sig, form:fcrt, from:'null'()}"]
	if {[regexp {vers\(([0-9]+)} $vers dum vers]} {
		return [string trimleft [string range $vers 0 1].[string range $vers 2 3] 0]
	}
	return 0
}

proc htmlCommentStrings {} {
	if {[htmlIsInContainer SCRIPT] || [htmlIsInContainer STYLE]} {
		return [list "/* " " */"]
	} else {
		return [list "<!-- " " -->"]
	}
}

# Create a string for URL mapping in Big Brother.
proc htmlURLmap {} {
	global HTMLmodeVars
	set urlmap {}
	foreach hp $HTMLmodeVars(homePages) {
		set fld "[htmlURLescape [lindex $hp 0] 1]/"
		regsub -all ":" $fld "/" fld
		set url [htmlURLescape "[lindex $hp 1][lindex $hp 2]"]
		lappend urlmap "Msta:$url, Mend:file:///$fld"
		append urlmap ","
	}
	set urlmap [string trimright $urlmap ","]
	return $urlmap
}

# Makes a line for browser error window.
proc htmlBrwsErr {fil l lnum ln text path} {
	return "$fil[format "%$l\s" ""]; Line $lnum:[format "%$ln\s" ""]$text\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t$path\r"
}

proc htmlSetWin {} {
	insertColorEscape 0 1
	insertColorEscape [nextLineStart [nextLineStart 0]] 0
	select [nextLineStart [nextLineStart 0]] [nextLineStart [nextLineStart [nextLineStart 0]]]
	setWinInfo dirty 0
	setWinInfo read-only 1
	scrollUpLine; scrollUpLine
	refresh
}
	
proc htmlIsTextFile {fil cmd} {
	if {[file isdirectory $fil] || [getFileType $fil] != "TEXT"} {
		$cmd "[file tail $fil] is not a text file."
		return 0
	}
	return 1
}

proc htmlAllSaved {msg} {
	set dirty 0
	foreach w [winNames] {
		getWinInfo -w $w arr
		if {$arr(dirty)} {set dirty 1; break}
	}
	if {$dirty} {
		set yn [eval [concat askyesno $msg]]
		if {$yn == "yes"} {saveAll}
		return $yn
	}
	return yes
}

# Determines in which home page folder a URL points to.
# If none, return empty string.
proc htmlInWhichHomePage {url} {
	global HTMLmodeVars
	foreach p $HTMLmodeVars(homePages) {
		if {[string match "[lindex $p 1][lindex $p 2]*" $url]} {return [lindex $p 0]}
	}
	return ""
}

# Asks for a folder and checks that it is not an alias.
proc htmlGetDir {prompt} {
	while {1} {
		if {[file isdirectory [set folder [get_directory -p $prompt]]]} {
			break
		} else {
			alertnote "Sorry! Cannot resolve aliases."
		}
	}
	return [string trimright $folder :]
}

proc htmlSetCase {elem} {
	global HTMLmodeVars 
	if {$HTMLmodeVars(useLowerCase)} { 
		return [string tolower $elem] 
	} else {
		return [string toupper $elem] 
	}
}


# Returns a list of all attributes used in any HTML element.
proc htmlGetAllAttrs {} {
	global htmlElemAttrOptional1 htmlElemAttrRequired1 htmlElemEventHandler1
	
	foreach elem [array names htmlElemAttrOptional1] {
		if {[info exists htmlElemAttrRequired1($elem)]} {
			append allHTMLattrs " " $htmlElemAttrRequired1($elem)
		}
		append allHTMLattrs " " $htmlElemAttrOptional1($elem)
		if {[info exists htmlElemEventHandler1($elem)]} {
			append allHTMLattrs " " [string toupper $htmlElemEventHandler1($elem)]
		}
	}
	return $allHTMLattrs
}


# Snatch the current selection into htmlCurSel, set flag whether there is one
proc htmlGetSel {} {
	global htmlCurSel htmlIsSel
	set htmlCurSel [string trim [getSelect]]
	set htmlIsSel [string length $htmlCurSel]
}


# Insert one or two carriage returns at the insertion point if any
# character preceding the insertion point (on the same line)
# is a non-whitespace character.
proc htmlOpenCR {indent {extrablankline 0}} {
	set end [getPos]
	set start [lineStart $end]
	set text [getText $start $end]
	if {![is::Whitespace $text]} {
		set r "\r$indent"
		if {$extrablankline} {append r "\r$indent"}
		return $r
	} elseif {$start > 0 } { 
		set prevstart [lineStart [expr $start - 1 ]]
		set text [getText $prevstart [expr $start - 1]]
		if {![is::Whitespace $text] && $extrablankline} {
			return "\r$indent"
		} else { 
			return [htmlFirstLineIndent $indent]
		}
	} else {
		return [htmlFirstLineIndent $indent]
	}
}

# Insert a carriage return at the insertion point if any
# character following the insertion point (on the same line)
# is a non-whitespace character.
proc htmlCloseCR {indent {start ""}} {
	if {$start == ""} {set start [selEnd]}
	if {![is::Whitespace [getText $start [nextLineStart $start]]]} {
		return "\r$indent"
	}
}

# Insert up to two carriage return at the insertion point depending
# on how many blank lines there are after the insertion point.
proc htmlCloseCR2 {indent pos} {
	set blank1 [is::Whitespace [getText $pos [nextLineStart $pos]]]
	set blank2 [is::Whitespace [getText $pos [nextLineStart [nextLineStart $pos]]]]
	if {!$blank1} {
		return "\r$indent\r$indent"
	} elseif {!$blank2} {
		return "\r$indent"
	}	
}

proc HTML::electricSemi {} {
	if {![htmlIsInContainer SCRIPT] && ![htmlIsInContainer STYLE]} {
		insertText ";"
		return
	}
	set pos [getPos]
	set start [lineStart $pos]
	set text [getText $start $pos]
	
	if {[string first "for" $text] != "-1"} {
		set lefts 0
		set rights 0
		set len [string length $text]
		for {set i 0} {$i < $len} {incr i} {
			case [string index $text $i] in {
				"("	{ incr lefts }
				")"	{ incr rights }
			}
		}
		if {$lefts != $rights} {
			insertText ";"
			return
		}
	}
	
	insertText ";\r" [htmlGetIndent $pos]
}

#===============================================================================
# Building tags, including element attributes
#===============================================================================

# A couple of functions to get element variables from the right package.
proc htmlGetSomeAttrs {item type num1} {
	global htmlElem${type}$num1
	if {[catch {set atts [set htmlElem${type}${num1}($item)]}]} { 
		set atts {} 
	}
	return $atts
}	

proc htmlGetRequired {item} {
	return [htmlGetSomeAttrs $item AttrRequired 1]
}

proc htmlGetOptional {item {all 0}} {
	set attrs [concat [htmlGetSomeAttrs $item AttrOptional 1] [htmlGetSomeAttrs $item EventHandler 1]]
	if {$all} {return $attrs}
	global HTMLmodeVars htmlHideDeprecated htmlHideExtensions
	set hidden [htmlGetHidden $item]
	set exp1 "\[ \n\r\t]+([join $HTMLmodeVars(alwaysaskforAttributes) |])"
	regsub -all $exp1 " $hidden" " " hidden
	set exp "\[ \n\r\t]+([join $HTMLmodeVars(dontaskforAttributes) |])"
	regsub -all $exp " $hidden" " " hidden
	set exp "\[ \n\r\t]+([join $hidden |])"
	regsub -all $exp " $attrs" " " attrs
	set exp "\[ \n\r\t]+([join $HTMLmodeVars(neveraskforAttributes) |])"
	regsub -all $exp " $attrs" " " attrs
	if {$htmlHideDeprecated || $HTMLmodeVars(hideDeprecated)} {
		set exp "\[ \n\r\t]+([join [concat [htmlGetExtensions $item] [htmlGetDeprecated $item]] |])"
		regsub -all $exp " $attrs" " " attrs
		if {$htmlHideDeprecated} {regsub "TARGET=" $attrs " " attrs}
	} elseif {$htmlHideExtensions || $HTMLmodeVars(hideExtensions)} {
		set exp "\[ \n\r\t]+([join [htmlGetExtensions $item] |])"
		regsub -all $exp " $attrs" " " attrs
	}
	return $attrs
}

proc htmlGetNumber {item} {
	return [htmlGetSomeAttrs $item AttrNumber 1]
}


proc htmlGetChoices {item} {
	return [htmlGetSomeAttrs $item AttrChoices 1]
}

proc htmlGetUsed {item {reqatts ""} {optatts ""} {arr 0}} {
	global HTMLmodeVars
	set useatts [htmlGetSomeAttrs $item AttrUsed ""]
	if {$arr} {return $useatts}
	if {$reqatts == ""} {set reqatts [htmlGetRequired $item]}
	if {$optatts == ""} {set optatts [htmlGetOptional $item]}
	set exp "\[ \n\r\t]+([join [concat $useatts $HTMLmodeVars(alwaysaskforAttributes)] |])"
	regsub -all $exp " $optatts" " " opt1
	set exp "\[ \n\r\t]+([join $opt1 |])"
	regsub -all $exp " $optatts" " " useatts
	return [concat $reqatts $useatts]
}

proc htmlGetHidden {item} {
	return [htmlGetSomeAttrs $item AttrHidden ""]
}

proc htmlGetExtensions {item} {
	return [htmlGetSomeAttrs $item Extension ""]
}

proc htmlGetDeprecated {item} {
	return [htmlGetSomeAttrs $item Deprecated ""]
}

proc htmlOpenElem {elem {used ""} {pos -1}} {
	global HTMLmodeVars 
	if {$HTMLmodeVars(useBigWindows)} {
		return [htmlOpenElemWindow $elem $used $pos]
	} else {
		return [htmlOpenElemStatusBar $elem $used $pos]
	}
}

# Opening or only tag of an element - include attributes
# Big window with all attributes.
# Return empty string if user clicks "Cancel".

proc htmlOpenElemWindow {elem used wrPos {values ""} {addNotUsed 0} {addHidden 0} {absPos ""}} {
	global HTMLmodeVars  htmlColorName htmlElemEventHandler1
	global  htmluserColors basicColors
	global htmlURLAttr htmlColorAttr  htmlWindowAttr
	global htmlSpecURL htmlSpecColor htmlSpecWindow
	
	set URLs $HTMLmodeVars(URLs)
	set Windows {_self _top _parent _blank}
	if {[llength $HTMLmodeVars(windows)]} {append Windows " - " $HTMLmodeVars(windows)}
	
# put users colours first
	set htmlColors [lsort [array names htmluserColors]]
 	append htmlColors " - " $basicColors

	if {![string length $used]} {set used $elem}
	set elem [string toupper $elem]
	set used [string toupper $used]
	
	# get variables for the element
	set reqatts [htmlGetRequired $used]
	set numatts [htmlGetNumber $used]
	set eventatts [htmlGetSomeAttrs $used EventHandler 1]
	set optatts [htmlGetOptional $used]
	set choiceatts [htmlGetChoices $used]
	set notUsedAtts ""
	set allatts [htmlGetUsed $used $reqatts $optatts]
	regsub -all "\[ \n\r\t]+([join $allatts |])" " $optatts" " " notUsedAtts
	if {$addNotUsed} {
		append allatts " $notUsedAtts"
		set notUsedAtts ""
	}
	if {$addHidden} {
		regsub -all "\[ \n\r\t]+([join $optatts |])" " [htmlGetOptional $used 1]" " " hiddenAtts
		append allatts " $hiddenAtts"
	}

	# if there are attributes to ask about, do so

	set text "<"
	append text  [htmlSetCase $elem]
	if {![llength $allatts]} {return "$text>"}

	set maxHeight [expr [lindex [getMainDevice] 3] - 115]
	set thisPage "Page 1"

	set widthIndex -1
	set heightIndex -1
	if {$absPos == ""} {set absPos [getPos]}
	# build window with attributes 
	set invalidInput 1
	while {$invalidInput} {
		# wrapping
		set htmlWrapPos [expr $wrPos == -1 ? [posX [getPos]] : $wrPos]
		incr htmlWrapPos [expr [string length $text] + 1]
		while {1} {
			if {$used == "LI IN UL" || $used == "LI IN OL"} {
				set pr LI
			} else {
				set pr $used
			}
			set box1 "-t {Attributes for $pr} 120 10 450 25"
			set box2 "-t {Attributes for $pr} 120 10 450 25"
			set box3 "-t {Attributes for $pr} 120 10 450 25"
			set page 1
			set attrtypes {}
			set fileIndex ""
			set colorIndex ""
			set wpos 10
			if {[string length $reqatts]} {
				lappend box$page -p 120 30 270 31 -t {Required attributes} 10 35 200 50
				set hpos 60
			} else {
				set hpos 30
			}
			set attrIndex 2
			for {set i 0} {$i < [llength $allatts]} {incr i} {
				set attr [lindex $allatts $i]
				if {$i == [llength $reqatts]} {
					if {$wpos > 20} { incr hpos 20 }
					lappend box$page -p 120 $hpos 270 [expr $hpos + 1] \
					-t {Optional attributes} 10 [expr $hpos + 5] 200 [expr $hpos + 20]
					set wpos 10
					incr hpos 30
				}
				set a2 [string trimright $attr =]
				if {[string index $attr [expr [string length $attr] - 1]] != "="}  { 
					# Flag
					if {[expr $hpos + 20] > $maxHeight && $wpos < 20 && $page < 3} {
						incr page
						set hpos 40
					}
					lappend box$page -c $attr [lindex $values $attrIndex] $wpos $hpos [expr $wpos + 100] [expr $hpos + 15]
					incr attrIndex 
					if {$wpos > 20} { 
						incr hpos 25
						set wpos 10
					} else {
						set wpos 230
					}
					lappend attrtypes flag
				} elseif {([lsearch -exact $htmlURLAttr $attr] >= 0 && [lsearch -exact $htmlSpecURL "${used}!=$a2"] < 0) || \
				[lsearch -exact $htmlSpecURL "${used}=$a2"] >= 0} { 
					# URL
					if {$wpos > 20} { incr hpos 25 ; set wpos 10}
					if {[expr $hpos + 45] > $maxHeight && $page < 3} {
						incr page
						set hpos 40
					}
					lappend box$page -t $attr 10 $hpos 120 [expr $hpos + 15] \
					-e [lindex $values $attrIndex] 120 $hpos 450 [expr $hpos + 15] \
					-m [concat [list [lindex $values [expr $attrIndex + 1]] {No value}] $URLs] \
					120 [expr $hpos + 25] 450 [expr $hpos + 35] \
					-b "File" 10 [expr $hpos + 20] 70 [expr $hpos + 40]
					incr attrIndex 3
					incr hpos 50
					lappend attrtypes url
					lappend fileIndex [expr $attrIndex - 1]
				} elseif {([lsearch -exact $htmlColorAttr $attr] >= 0 && [lsearch -exact $htmlSpecColor "${used}!=$a2"] < 0) || \
				[lsearch -exact $htmlSpecColor "${used}=$a2"] >= 0} { 
					# Color attribute
					if {$wpos > 20} { incr hpos 25 ; set wpos 10}					
					if {[expr $hpos + 25] > $maxHeight && $page < 3} {
						incr page
						set hpos 40
					}
					lappend box$page -t $attr 10 $hpos 120 [expr $hpos + 15] \
					-e [lindex $values $attrIndex] 120 $hpos 190 [expr $hpos + 15] \
					-m [concat [list [lindex $values [expr $attrIndex + 1]] {No value}] $htmlColors] \
					200 $hpos 340 [expr $hpos + 15] \
					-b "New Color" 350 $hpos 450 [expr $hpos + 20]
					incr attrIndex 3
					incr hpos 30
					lappend attrtypes color
					lappend colorIndex [expr $attrIndex - 1]
				} elseif {([lsearch -exact $htmlWindowAttr $attr] >= 0 && [lsearch -exact $htmlSpecWindow "${used}!=$a2"] < 0) || \
				[lsearch -exact $htmlSpecWindow "${used}=$a2"] >= 0} { 
					# Window attribute
					if {$wpos > 20} { incr hpos 25 ; set wpos 10}					
					if {[expr $hpos + 25] > $maxHeight && $page < 3} {
						incr page
						set hpos 40
					}
					lappend box$page -t $attr 10 $hpos 120 [expr $hpos + 15] \
					-e [lindex $values $attrIndex] 120 $hpos 240 [expr $hpos + 15] \
					-m [concat [list [lindex $values [expr $attrIndex + 1]] {No value}] \
					$Windows] \
					250 $hpos 440 [expr $hpos + 15]
					incr attrIndex 2
					incr hpos 30
					lappend attrtypes window
				} elseif {[lsearch $numatts "${attr}*"] >= 0} { 
					# Number
					if {[expr $hpos + 20] > $maxHeight && $wpos < 20 && $page < 3} {
						incr page
						set hpos 40
					}
					if {$attr == "WIDTH="} {set widthIndex $attrIndex}
					if {$attr == "HEIGHT="} {set heightIndex $attrIndex}
					lappend box$page -t $attr $wpos $hpos [expr $wpos + 100] [expr $hpos + 15] \
					-e [lindex $values $attrIndex] [expr $wpos + 110] $hpos [expr $wpos + 150] [expr $hpos + 15]
					incr attrIndex 
					if {$wpos > 20} { 
						incr hpos 25
						set wpos 10
					} else {
						set wpos 230
					}
					lappend attrtypes number
				} elseif {[lsearch $choiceatts "${attr}*"] >= 0} { 
					# Choices
					if {[expr $hpos + 20] > $maxHeight && $wpos < 20 && $page < 3} {
						incr page
						set hpos 40
					}
					set matches {}
					foreach w $choiceatts {
						if {[string match "${attr}*" $w]} {
							lappend matches  [string range $w [string length $attr] end]
						}	
					}
					lappend box$page -t $attr $wpos $hpos [expr $wpos + 100] [expr $hpos + 15] \
					-m [concat [list [lindex $values $attrIndex] {No value}] $matches] \
					[expr $wpos + 110] $hpos [expr $wpos + 205] [expr $hpos + 15]
					incr attrIndex 
					if {$wpos > 20} { 
						incr hpos 25 
						set wpos 10
					} else {
						set wpos 230
					}	
					lappend attrtypes choices
				} else {
					# Any other
					if {$wpos > 20} { incr hpos 25 ; set wpos 10}					
					if {[expr $hpos + 20] > $maxHeight && $page < 3} {
						incr page
						set hpos 40
					}
					lappend box$page -t $attr 10 $hpos 120 [expr $hpos + 15] \
					-e [lindex $values $attrIndex] 120 $hpos 450 [expr $hpos + 15] 
					incr attrIndex
					incr hpos 25
					lappend attrtypes any
				}
			}
			if {$wpos > 20} { incr hpos 25 }
			
			if {$page == 1} {
				set box $box1
			} elseif {$page == 2} {
				set hpos $maxHeight
				set box " -m \{\{$thisPage\} \{Page 1\} \{Page 2\}\} 10 10 85 30 -n \{Page 1\} $box1 -n \{Page 2\} $box2"
			} elseif {$page == 3} {
				set hpos $maxHeight
				set box " -m \{\{$thisPage\} \{Page 1\} \{Page 2\} \{Page 3\}\} 10 10 85 30 -n \{Page 1\} $box1 -n \{Page 2\} $box2 -n \{Page 3\} $box3"
			}
			# Add More button if hidden attrs
			set moreButton 0
			if {[llength $notUsedAtts]} {
				set box " -b More 200 [expr $hpos + 20] 265 [expr $hpos + 40] $box"
				set moreButton 1
			}
			set values [eval [concat dialog -w 460 -h [expr $hpos + 50] \
			-b OK 20 [expr $hpos + 20]  85 [expr $hpos + 40] \
			-b Cancel 110 [expr $hpos + 20] 175 [expr $hpos + 40] $box]]
			
			# More button clicked?
			if {$moreButton && [lindex $values 2]} {
				append allatts " $notUsedAtts"
				set notUsedAtts ""
			}
			# If more button...
			if {$moreButton} {
				set values [lreplace $values 2 2]
			}
			# If two pages...
			if {$page > 1} {
				set thisPage [lindex $values 2]
				set values [lreplace $values 2 2]
			}
			# OK button clicked?
			if {[lindex $values 0] } { break }
			# Cancel button clicked?
			if {[lindex $values 1] } { return}
			# File button clicked?
			foreach fl $fileIndex {
				if {[lindex $values $fl] && [string length [set newFile [htmlGetFile]]]} {
					set URLs $HTMLmodeVars(URLs)
					set values [lreplace $values [expr $fl - 1] [expr $fl - 1] [lindex $newFile 0]]
					if {$used == "IMG" && $fl == 4 && [llength [set widhei [lindex $newFile 1]]]} {
						if {$widthIndex >= 0} {set values [lreplace $values $widthIndex $widthIndex [lindex $widhei 0]]}
						if {$heightIndex >= 0} {set values [lreplace $values $heightIndex $heightIndex [lindex $widhei 1]]}
					}
				}
			}
			# Color button clicked?
			foreach cl $colorIndex {
				if {[lindex $values $cl] && [string length [set newcolor [htmlAddNewColor]]]} {
					set htmlColors [concat [list $newcolor] $htmlColors]
					set values [lreplace $values [expr $cl - 1] [expr $cl - 1] "$newcolor"]
				}
			}
		}
		

		# put everything together
		set attrtext ""
		set errtext ""

		set j 2
		for {set i 0} {$i < [llength $attrtypes]} {incr i} {
			set attr [lindex $allatts $i]				
			switch [lindex $attrtypes $i] {
				url {
					set texturl [string trim [lindex $values $j]]
					set menuurl [lindex $values [expr $j + 1]]
					if {[string length $texturl]} {		
						append attrtext [htmlWrapTag "[htmlSetCase $attr][htmlAddQuotes [htmlURLescape2 $texturl]]"]
						htmlAddToCache URLs $texturl
					} elseif {$menuurl != "No value"} {
						append attrtext [htmlWrapTag "[htmlSetCase $attr][htmlAddQuotes [htmlURLescape2 $menuurl]]"] 
					} elseif {[lsearch -exact $reqatts $attr] >= 0} {
						lappend errtext "$attr required."
					}
					incr j 3
				}
				color {
					set colortxt [lindex $values $j]
					set colorval [lindex $values [expr $j + 1]]
					if {[string length $colortxt]} {
						set col [htmlCheckColorNumber $colortxt]
 								if {$col == 0} {
 									lappend errtext "$attr: $colortxt is not a valid color number."
						} else {	
							append attrtext [htmlWrapTag "[htmlSetCase $attr][htmlAddQuotes $col]"]
						}
					} elseif {$colorval != "No value"} {
						# Users own color?
						if {[info exists htmluserColors($colorval)]} {
							set colornum $htmluserColors($colorval)
						}
						# Predefined color?
						if {[info exists htmlColorName($colorval)]} {
							set colornum $htmlColorName($colorval)
						}
						append attrtext [htmlWrapTag "[htmlSetCase $attr][htmlAddQuotes $colornum]"]
					} elseif {[lsearch -exact $reqatts $attr] >= 0} {
						lappend errtext "$attr required."
					}
					incr j 3
				}
				window {
					set textwin [string trim [lindex $values $j]]
					set menuwin [lindex $values [expr $j + 1]]
					if {[string length $textwin]} {		
						append attrtext [htmlWrapTag "[htmlSetCase $attr][htmlAddQuotes $textwin]"]
						htmlAddToCache windows $textwin
					} elseif {$menuwin != "No value"} {
						append attrtext [htmlWrapTag "[htmlSetCase $attr][htmlAddQuotes $menuwin]"]
					} elseif {[lsearch -exact $reqatts $attr] >= 0} {
						lappend errtext "$attr required."
					}
					incr j 2
				}
				number {
					set numval [string trim [lindex $values $j]]
					if {[string length $numval]} {
						if {[htmlCheckAttrNumber $used $attr $numval] == 1} {		
							append attrtext [htmlWrapTag "[htmlSetCase $attr][htmlAddQuotes $numval]"]
						} else {
							lappend errtext "$attr: [htmlCheckAttrNumber $used $attr $numval]"
						}
					} elseif {[lsearch -exact $reqatts $attr] >= 0} {
						lappend errtext "$attr required."
					}
					incr j
				}
				choices {
					set choiceval [lindex $values $j]
					if {$choiceval != "No value"} {		
						set qchoice [htmlAddQuotes $choiceval]
						if {($used != "LI IN OL" && $used != "OL") || $attr != "TYPE="} {
							set qchoice [htmlSetCase $qchoice]
						}
						append attrtext [htmlWrapTag "[htmlSetCase $attr]$qchoice"]
					} elseif {[lsearch -exact $reqatts $attr] >= 0} {
						lappend errtext "$attr required."
					}
					incr j
				}
				any {
					set anyval [lindex $values $j]
					# Trim only if it's only spaces.
					if {[string trim $anyval] == ""} {set anyval ""}
					if {[string length $anyval]} {
						htmlOpenExtraThings $used $attr $anyval
						if {[lsearch -exact $eventatts $attr] < 0} {
							set attr [htmlSetCase $attr]
						}
						append attrtext [htmlWrapTag "$attr[htmlAddQuotes $anyval]"]
					} elseif {[lsearch -exact $reqatts $attr] >= 0} {
						lappend errtext "$attr required."
					}
					incr j
				}
				flag {
					set flagval [lindex $values $j]
					if {$flagval} {		
						append attrtext [htmlWrapTag [htmlSetCase $attr]]
					}
					incr j
				}
			}
		}	
		# If everything is OK, add the attribute text to text.
		if {![llength $errtext]} {
			append text $attrtext
			set invalidInput 0
		} else {
			# Put up alert with the error text.
			htmlErrorWindow "Invalid input for $used" $errtext
		}
		# Some tests that input is ok.
		if {!$invalidInput} {set invalidInput [htmlFontBaseTest $text alertnote]}
		if {!$invalidInput && $elem == "A" && [set invalidInput [htmlATest $text alertnote]]} {
			set text "<[htmlSetCase A]"
		}
		if {!$invalidInput && $elem == "FRAMESET" && [set invalidInput [htmlFramesetTest $text alertnote]]} {
			set text "<[htmlSetCase FRAMESET]"
		}
		if {!$invalidInput && $elem == "SPACER" && [set invalidInput [htmlSpacerTest $text alertnote]]} {
			set text "<[htmlSetCase SPACER]"
		}
		if {!$invalidInput && $elem == "AREA" && [set invalidInput [htmlAreaTest $text alertnote]]} {
			set text "<[htmlSetCase AREA]"
		}
	}
	
	if {[string length $text] } {append text ">"}
	
	return ${text}
}

proc htmlWrapTag {toadd} {
	global fillColumn HTMLmodeVars
	upvar htmlWrapPos wrpos absPos ap
	if {!$HTMLmodeVars(wordWrap)} {return " $toadd"}
	incr wrpos [string length $toadd]
	if {$wrpos > $fillColumn} {
		set ind [htmlGetIndent $ap]
		set wrpos [string length "$ind$toadd"]
		return "\r$ind$toadd"
	} else {
		return " $toadd"
	}
}

# these two require at least one of several optional attributes
proc htmlFontBaseTest {text cmd} {
	if {[string toupper $text] == "<FONT" || [string toupper $text] == "<BASEFONT" ||
	[string toupper $text] == "<BASE" || [string toupper $text] == "<SPAN"} {  
		eval {$cmd "At least one of the attributes is required."}
		return 1
	}
	return 0
}

# HREF or NAME must be used for A.
proc htmlATest {text cmd} {
	if {![regexp -nocase {href=} $text] && ![regexp -nocase {name=} $text]} {
		eval {$cmd "At least one of the attributes HREF and NAME must be used."}
		return 1
	}
	return 0
}

# ROWS or COLS must be used for FRAMESET
proc htmlFramesetTest {text cmd} {
	if {![regexp -nocase {rows=} $text] && ![regexp -nocase {cols=} $text]} {
		eval {$cmd "At least one of the attributes ROWS and COLS must be used."}
		return 1
	}
	return 0
}

# Some checks for SPACER.
proc htmlSpacerTest {text cmd} {
		set horver [regexp -nocase {type=\"(horizontal|vertical)\"} $text]
		set wh [regexp -nocase {width=|height=} $text]
		set sz [regexp -nocase {size=} $text]
		set al [regexp -nocase {align=} $text]
		set invalidInput 1
		if {$horver && ($wh || $al)} {
			eval {$cmd "WIDTH, HEIGHT and ALIGN should only be used when TYPE=BLOCK."}
		} elseif {!$horver && $sz} {
			eval {$cmd "SIZE should only be used when TYPE=HORIZONTAL or VERTICAL."}
		} elseif {$horver && !$sz} {
			eval {$cmd "SIZE is required when TYPE=HORIZONTAL or VERTICAL."}
		} elseif {!$horver && !$wh} {
			eval {$cmd "WIDTH or HEIGHT is required when TYPE=BLOCK."}
		} else {
			set invalidInput 0
		}
		return $invalidInput
}

# For AREA, either HREF or NOHREF must be used, but not both.
proc htmlAreaTest {text cmd} {
	set hasHref [regexp -nocase {href=} $text]
	set hasNohref [regexp -nocase {nohref} $text]
	set hasCoords [regexp -nocase {coords=} $text]
	set shapeDefault [regexp -nocase {shape=\"default\"} $text]
	set invalidInput 0
	if {($hasHref && $hasNohref) || (!$hasHref && !$hasNohref)} {
		eval {$cmd "One of the attributes HREF and NOHREF must be used, but not both."}
		set invalidInput 1
	} elseif {!$hasCoords && !$shapeDefault} {
		eval {$cmd "COORDS= is required if SHAPEDEFAULT"}
		set invalidInput 1
	}
	return $invalidInput
}

# Adds a NAME= value to cache.
proc htmlOpenExtraThings {elem attr val} {
	if {[lsearch -exact {A MAP} $elem] >= 0 && $attr == "NAME="} {
		htmlAddToCache URLs "#$val"
	}
	if {$elem == "FRAME" && $attr == "NAME="} {
		htmlAddToCache windows $val
	}
}


# Check if a input is a valid number for the element attribute.
# Returns 1 if it is, otherwise returns an error message.
proc htmlCheckAttrNumber {item attr number} {
	
	set attrNumbers [htmlGetNumber $item]
	set numind [lsearch $attrNumbers "${attr}*"]
	set numstr [string range [lindex $attrNumbers $numind] [string length $attr] end]
	regexp {^[-i0-9]+} $numstr minvalue
	set numstr [string range $numstr [expr [string length $minvalue] + 1] end]
	regexp {^[-i0-9]+} $numstr maxvalue
	set procent [string range $numstr [expr [string length $numstr] - 1] end]
	if {$procent == "%"} {
		set procerr " or percentage"
	} else {
		set procerr ""
	}
	if {$minvalue == "-i"} {
		set errtext "An integer"
	} elseif {$maxvalue == "i"} {
		set errtext "A number $minvalue or greater"
	} else {
		set errtext "A number in the range $minvalue to $maxvalue"
	}
	if {$item == "FONT"} { append errtext " or -6 to +6"}
	append errtext  "$procerr expected." 
	# Is percent allowed?
	if {[string index $number [expr [string length $number] - 1]] == "%" } {
		set number [string range $number 0 [expr [string length $number] - 2]]
		if {$procent != "%"} {return $errtext}
	}
	# FONT can take values -6 - +6. Special case.
	if {$item == "FONT" && [regexp {^(\+|-)[1-6]$} $number]} { return 1}
	# Is input a number?
	if {![regexp {^-?[0-9]+$} $number]} {return $errtext}
	# Is input in the valid range?
	if {( $maxvalue != "i" && $number > $maxvalue ) || ( $minvalue != "-i" && $number < $minvalue ) } {
		return $errtext
	}	
	return 1 
}


# Add quotes to attribute
proc htmlAddQuotes {v} {

	if {[string range $v 0 0] != "\""} {set v  "\"$v"}
 	set vlen [expr [string length $v] - 1]
	if {[string range $v $vlen $vlen] !="\""} {append v "\""}
	return $v
}


# Splits an attribute into its name and value and remove quotes.
proc htmlRemoveQuotes {attrStr} {
	# Is it a flag?
	if {![string match "*=*" $attrStr]} {return [string toupper $attrStr]}
	
	set attr [string range $attrStr 0 [string first "=" $attrStr]]
	# Get the attribute value.
	set attrVal [string range $attrStr [expr [string first "=" $attrStr] + 1] end]
	
	return [list $attr [string trim $attrVal \"]]
}

# Returns a list of the attributes not used for the tag at the current position.
proc htmlGetAttributes {} {
	global htmlElemKeyBinding
	set pos [getPos]
	if {[catch {search -s -f 0 -r 1 -m 0 {<[^<>]+>} $pos} res] || [lindex $res 1] < $pos} {
		message "Current position is not at a tag."
		return
	}
	set tag [string trim [lindex [set all [string toupper [eval getText $res]]] 0] "<>"]
	if {$tag == "LI"} {
		set ltype [htmlFindList]
		if {$ltype == "UL"} {
			set tag "LI IN UL"
		} elseif {$ltype == "OL"} {
			set tag "LI IN OL"
		}			
	}
	# All INPUT elements are defined differently. Must extract TYPE.
	if {$tag == "INPUT"} {
		if {![regexp -nocase { TYPE=\"?([^ \t\r\"<>]+)\"?} $all dum tag]} {
			message "INPUT element without a TYPE attribute."
			return
		}
		set tag [string toupper $tag]
		if {![info exists htmlElemKeyBinding($tag)]} {set tag "INPUT TYPE=$tag"}
	}
	set ret ""
	foreach a [concat [htmlGetRequired $tag] [htmlGetOptional $tag]] {
		set exp "\[ \t\r\n\]+${a}"
		if {![regexp -nocase $exp $all]} {
			lappend ret $a
		}
	}
	if {$ret == ""} {message "No attributes."}
	return $ret
}

# Inserts an attribute in a tag at the current position.
proc htmlInsertAttributes {{attrList ""}} {
	global HTMLmodeVars fillColumn elecStopMarker
	set useMarks $HTMLmodeVars(useTabMarks)
	if {$attrList == "" && ([set l [htmlGetAttributes]] == "" ||
	[catch {listpick -p "Select attributes" -l $l} attrList] || $attrList == "") } {return}
	foreach attr $attrList {
		set epos [expr [lindex [search -s -f 0 -r 1 -m 0 {<[^<>]+>} [getPos]] 1] - 1]
		if {[expr [posX $epos] + [string length $attr]] > $fillColumn && $HTMLmodeVars(wordWrap)} {
			set text "\r[htmlGetIndent $epos]"
		} else {
			set text " "
		}
		append text $attr
		if {[string match "*=" $attr]} {
			append text "\"\""
			if {$useMarks} {append text $elecStopMarker}		
		}
		set x [expr $epos - 3]
		if {[string match "*$elecStopMarker" [set etxt [getText $x $epos]]]} {
			set p [expr $x + 1]
			if {$useMarks} {
				if {[string match "*=" $attr]} {
					set text [string range $text 0 [expr [string length $text] - 3]]$elecStopMarker\"$elecStopMarker
				} else {
					append text $elecStopMarker
				}
			}
			replaceText [expr $p + 1] $epos $text
		} else {
			goto $epos
			insertText $text
			if {[regexp {=} $text]} {goto [expr + [getPos] - 1 - $useMarks]}
		}
	}
}

#===============================================================================
# Element build routines
#===============================================================================

# Closing tag of an element
proc htmlCloseElem {theElem} {
	return "</[htmlSetCase $theElem]>"
}


proc htmlTag {str} {
	global htmlElemProc
	set elem [lindex $str 1]
	if {[htmlIsInContainer STYLE]} {
		if {[lindex $str 0] == "htmlBuildInputElem"} {set elem INPUT}
		replaceText [getPos] [selEnd] $elem
	} elseif {[info exists htmlElemProc($elem)]} {
		eval $htmlElemProc($elem)
	} else {
		eval $str
	}
}

# Build elements with only a opening tag.
proc htmlBuildOpening {ftype {begCR 0} {endCR 0} {attr ""}} {
	set text1 ""
	set indent [htmlGetIndent [getPos]]
	if {$begCR} { 
		set text1 [htmlOpenCR $indent]
	}
	set text [htmlOpenElem $ftype $attr]
	if {![string length $text]} {return}
	if {$endCR} {
		append text [htmlCloseCR $indent]
	}
	insertText $text1 $text
}

	
# This is used for almost all containers
proc htmlBuildElem {ftype {attr ""}} {
	global HTMLmodeVars htmlCurSel htmlIsSel elecStopMarker

	if {![string length [set text [htmlOpenElem $ftype $attr]]]} {return}
	htmlGetSel
	append text $htmlCurSel
	set currpos [expr [getPos] + [string length $text]]
	append text [htmlCloseElem $ftype]
	if {!$htmlIsSel && $HTMLmodeVars(useTabMarks)} {append text $elecStopMarker}
	if {$htmlIsSel} {
		replaceText [getPos] [selEnd] $text
	} else {
		insertText $text
		goto $currpos
	}
}

# This is used for elements that should be surrounded by newlines
proc htmlBuildCRElem {ftype {extrablankline 0} {attr ""}} {
	global htmlCurSel htmlIsSel HTMLmodeVars elecStopMarker

	if {![string length [set text2 [htmlOpenElem $ftype $attr 0]]]} {return}
	set indent [htmlFindNextIndent]
	set text [htmlOpenCR $indent $extrablankline]
	append text $text2
	htmlGetSel
	append text $htmlCurSel
	set currpos [expr [getPos] + [string length $text]]
	append text [htmlCloseElem $ftype]
	if {$extrablankline} {
		set cr2 [htmlCloseCR2 $indent [selEnd]]
	} else {
		set cr2 [htmlCloseCR $indent]
	}
	append text $cr2
	if {!$htmlIsSel && $HTMLmodeVars(useTabMarks)} {append text $elecStopMarker}
	if {$htmlIsSel} { deleteSelection }
	insertText $text
	if {!$htmlIsSel} {
		goto $currpos
	}
}

# This is used for elements that should be surrounded by empty lines
proc htmlBuildCR2Elem {ftype {attr ""}} {
	global HTMLmodeVars htmlCurSel htmlIsSel elecStopMarker indentationAmount
	
	htmlGetSel
# Check if user has skipped an attribute which can't be skipped.
	if {![string length [set text2 [htmlOpenElem $ftype $attr 0]]]} {return}
	set indent [htmlFindNextIndent]
	set text [htmlOpenCR $indent 1]
	append text $text2
	if {[info exists HTMLmodeVars(indent${ftype})] && $HTMLmodeVars(indent${ftype})} {
		set exindent [text::maxSpaceForm [text::Tab]]
		htmlIndentChunk htmlCurSel
	} else {
		set exindent ""
	}
	if {$htmlIsSel || ($ftype != "SCRIPT" && $ftype != "STYLE")} {
		append text "\r" [text::minSpaceForm "${indent}${exindent}"] $htmlCurSel
	} else {
		append text "\r${indent}<!-- /* Hide content from old browsers */\r${indent}"
	}
	set currpos [expr [getPos] + [string length $text]]
	append text \r$indent
	set pre(SCRIPT) "//"; set pre(STYLE) "/*"; set post(SCRIPT) ""; set post(STYLE) "*/"
	if {!$htmlIsSel && ($ftype == "SCRIPT" || $ftype == "STYLE")} {append text "$pre($ftype) end hiding content from old browsers $post($ftype) -->\r$indent"}
	append text [htmlCloseElem $ftype]
	append text [htmlCloseCR2 $indent [selEnd]]
	if {!$htmlIsSel && $HTMLmodeVars(useTabMarks)} {append text $elecStopMarker}
	if {$htmlIsSel} { deleteSelection }
	insertText $text
	if {!$htmlIsSel}	{
		goto $currpos
	}
}

# Determines which list the current position is inside.
proc htmlFindList {} {	
	set listType ""
	foreach l [list UL OL DIR MENU] {
		set ex "<${l}(\[ \\t\\r\]+\[^>\]*>|>)"
		set listOpening [search -s -f 0 -i 1 -r 1 -m 0 -n $ex [getPos]]
		set ex2 </$l>
		set listClosing [search -s -f 0 -i 1 -r 1 -m 0 -n $ex2 [getPos]]
		# Search until a single list opening is found.
		while {[string length $listOpening] && [string length $listClosing] &&
		[lindex $listClosing 0] > [lindex $listOpening 0]} {
			set listOpening [search -s -f 0 -i 1 -r 1 -m 0 -n $ex [expr [lindex $listOpening 0] - 1]]
			set listClosing [search -s -f 0 -i 1 -r 1 -m 0 -n $ex2 [expr [lindex $listClosing 0] - 1]]
		}
		if {[string length $listOpening]} {
			lappend listType "$listOpening $l"
		}
	}
	set ltype [lindex [lindex $listType 0] 2]
	set lnum [lindex [lindex $listType 0] 0]
	for {set i 1} {$i < [llength $listType]} {incr i} {
		if {[lindex [lindex $listType $i] 0] > $lnum} {
			set ltype [lindex [lindex $listType $i] 2]
			set lnum [lindex [lindex $listType $i] 0]
		}
	}
	return $ltype
}


# Choose an item from Use Attributes menu.
proc htmlUseAttributes {} {
	global htmlElemAttrOptional1
	foreach a [array names htmlElemAttrOptional1] {
		if {[llength $htmlElemAttrOptional1($a)]} {lappend htmlPossibleToUse $a}
	}
	regsub " S " $htmlPossibleToUse " " htmlPossibleToUse
	if {![catch {listpick -p "Choose HTML element" [lsort $htmlPossibleToUse]} elem] &&
	$elem != ""} {htmlUseAttributes2 $elem}
}

# Customize list of attributes which get asked about
proc htmlUseAttributes2 {item} {
	global htmlElemAttrUsed htmlElemAttrHidden
	set reqattrs [htmlGetRequired $item]
	set optatts [htmlGetOptional $item 1]
	set used [htmlGetUsed $item $reqattrs $optatts 1]
	set extensions [htmlGetExtensions $item]
	set hidden [htmlGetHidden $item]
	htmlUseAttrsDialog "Attributes for $item" $optatts $extensions used hidden
	set htmlElemAttrUsed($item) $used
	set htmlElemAttrHidden($item) $hidden
	addArrDef htmlElemAttrUsed $item $used
	addArrDef htmlElemAttrHidden $item $hidden
}

proc htmlUseAttrsDialog {txt optatts extensions us hi {do ""} {isGlobal 0}} {
	global HTMLmodeVars modifiedModeVars
	upvar $us used $hi hidden
	if {$do != ""} {upvar $do dont}
	set hideExtensions $HTMLmodeVars(hideExtensions)
	set hideDeprecated $HTMLmodeVars(hideDeprecated)
	set alwaysask $HTMLmodeVars(alwaysaskforAttributes)
	set dontask $HTMLmodeVars(dontaskforAttributes)
	set neverask $HTMLmodeVars(neveraskforAttributes)
	set page 0
	set attrnumber [llength $optatts]
	set options {"Always ask about" "Don't ask about at first" "Never ask about"}
	set len 10
	if {$isGlobal} {
		set options "{Use individual settings} $options"
		set len 8
	}
	foreach a $optatts {
		if {[lsearch -exact $used $a] >= 0} {
			lappend uh $isGlobal
		} elseif {[lsearch -exact $hidden $a] >= 0} {
			lappend uh [expr 2 + $isGlobal]
		} elseif {!$isGlobal || [lsearch -exact $dont $a] >= 0} {
			lappend uh [expr 1 + $isGlobal]
		} else {
			lappend uh 0
		}
	}
	while {1} {
		set box "-t [list $txt] 100 10 370 25"
		if {!$isGlobal} {append box " -t {Global settings} 380 10 540 25"}
		set h 35
		if {$isGlobal && !$page} {
			append box " -c {Don't use extensions to HTML 4.0} $hideExtensions 10 $h 370 [expr $h + 20]"
			append box " -c {Don't use deprecated elements and attributes} $hideDeprecated 10 [expr $h + 25] 370 [expr $h + 45]"
			incr h 50
		}
		set n 0
		foreach a [lrange $optatts [expr $len * $page] [expr $len * $page + $len - 1]] {
			set m [lindex $uh [expr $len * $page + $n]]
			append box " -t [string trimright $a =] 10 $h 150 [expr $h + 20] -m {[lrange $options $m $m] $options} 160 $h 370 [expr $h + 20]"
			if {!$isGlobal} {
				if {$hideExtensions && [lsearch -exact $extensions $a] >= 0 || [lsearch -exact $neverask $a] >= 0} {append box " -t {Never ask about} 380 $h 540 [expr $h + 20]"}
				if {[lsearch -exact $dontask $a] >= 0} {append box " -t {Don't ask about at first} 380 $h 540 [expr $h + 20]"}
				if {[lsearch -exact $alwaysask $a] >= 0} {append box " -t {Always ask about} 380 $h 540 [expr $h + 20]"}
			}
			incr h 25
			incr n
		}
		incr h 10
		set h1 [expr $h + 20]
		if {$page > 0} {append box " -b {<< Prev} 200 $h 265 $h1"}
		if {[expr $len * $page + $len] < $attrnumber} {append box " -b {Next >>} 290 $h 355 $h1"}
		set values [eval [concat dialog -w [expr $isGlobal ? 380 : 550] -h [expr $h + 30] -b OK 20 $h 85 $h1 -b Cancel 110 $h 175 $h1 $box]]
		if {$isGlobal && !$page} {
			set hideExtensions [lindex $values 2]
			set hideDeprecated [lindex $values 3]
			set values [lreplace $values 2 3]
		}
		if {[lindex $values 1]} {error "Cancel"}
		set uh1 ""
		foreach v [lrange $values 2 [expr $n + 1]] {
			lappend uh1 [lsearch -exact $options $v]
		}
		set uh [eval [concat lreplace [list $uh] [expr $len * $page] [expr $len * $page + $n - 1] $uh1]]
		if {[lindex $values 0]} {break}
		if {$page > 0 && [lindex $values [expr $n + 2]]} {
			incr page -1
		} else {
			incr page
		}
	}
	set used ""
	set hidden ""
	set dont ""
	for {set i 0} {$i < $attrnumber} {incr i} {
		if {[lindex $uh $i] == $isGlobal} {lappend used [lindex $optatts $i]}
		if {$isGlobal && [lindex $uh $i] == 2} {lappend dont [lindex $optatts $i]}
		if {[lindex $uh $i] == [expr 2 + $isGlobal]} {lappend hidden [lindex $optatts $i]}
	}
	foreach h {hideExtensions hideDeprecated} {
		if {[set $h] != $HTMLmodeVars($h)} {
			set HTMLmodeVars($h) [set $h]
			lappend modifiedModeVars [list $h HTMLmodeVars]
			htmlHide
		}
	}
}

#===============================================================================
# Indentation
#===============================================================================

proc HTML::indentLine {} {
	if {[htmlIsInContainer STYLE] || [htmlIsInContainer SCRIPT]} {text::genericIndent; return}
	if {[htmlIsInContainer PRE]} {return}
	
	set previndent [htmlFindIndent]
	set lend [expr [nextLineStart [getPos]] - 1]
	if {$lend < [getPos]} {set lend [maxPos]}
	set thisLine [string trimleft [getText [set lstart [lineStart [getPos]]] $lend ]]
	set thisIndent [htmlGetIndent [getPos]]
	if {$thisIndent != $previndent} {replaceText $lstart $lend "$previndent$thisLine"}

}

# Find the indentation the current line should have.
proc htmlFindIndent {{pos0 ""}} {
	global htmlIndentElements HTMLmodeVars
	set indent ""
	foreach i $htmlIndentElements {
		if {$HTMLmodeVars(indent$i)} {lappend indent $i}
	}
	# Find previous non-blank line.
	if {$pos0 == ""} {set pos0 [getPos]}
	set pos [expr [lineStart $pos0] - 1]
	while {$pos >= 0 && [regexp {^[ \t]*$} [getText [lineStart $pos] $pos]]} {
		set pos [expr [lineStart $pos] - 1]
	}
	set pos [expr $pos >= 0 ? $pos : 0]
	# Get indentation on that line.
	set previndent [htmlGetIndent $pos]
	# Find last tag on or before that line.
	if {[catch {search -s -f 0 -m 0 -r 1 {<([^<>]+)>} $pos} tag] || [lindex $tag 1] < [lineStart $pos] ||
	( [lindex $tag 0] < [lineStart $pos0] && [lindex $tag 1] > [lineStart $pos0])} {
		set tag ""
	} else {
		set tag [string trim [eval getText $tag] "<>"]
	}
	set tag [string toupper [lindex $tag 0]]
	# Add to indentation?
	if {[lsearch -exact $indent $tag] >= 0} {
		set previndent [htmlIncreaseIndent $previndent]
	}
	# Find last tag on current line.
	set tag ""
	set lstart [lineStart $pos0]
	set lend [expr ([set npos [nextLineStart $pos0]] <= $lstart) ? $lstart : $npos - 1]
	regexp {<([^<>]+)>} [getText $lstart $lend] dum tag
	set tag [string toupper [lindex $tag 0]]
	
	# Reduce indentation?
	if {[string index $tag 0] == "/" && [lsearch -exact $indent [string range $tag 1 end]] >= 0} {
		set previndent [htmlReduceIndent $previndent]
	}
	return $previndent 
}

# Find the indentation the next line should have.
proc htmlFindNextIndent {{pos0 ""}} {
	global HTMLmodeVars htmlIndentElements
	set indent ""
	foreach i $htmlIndentElements {
		if {$HTMLmodeVars(indent$i)} {lappend indent $i}
	}
	if {$pos0 == ""} {set pos0 [getPos]}
	set ind [htmlFindIndent $pos0]
	# Find last tag before pos0 on current line.
	set tag ""
	set lstart [lineStart $pos0]
	regexp {<([^<>]+)>} [getText $lstart $pos0] dum tag
	set tag [string toupper [lindex $tag 0]]
	# Add to indentation?
	if {[lsearch -exact $indent $tag] >= 0} {set ind [htmlIncreaseIndent $ind]}
	return $ind
}

# get the leading whitespace of the current line
proc htmlGetIndent { pos } {
	set res [search -s -n -f 1 -r 1 "^\[ \t\]*" [lineStart $pos]]
	return [text::minSpaceForm [eval getText $res]]
}

# Adds indentationAmount whitespace.
proc htmlIncreaseIndent {indent} {
	global indentationAmount
	set in [string range "                               " 1 $indentationAmount]
	return [text::minSpaceForm "[text::maxSpaceForm $indent]$in"]
}

# Removes indentationAmount whitespace.
proc htmlReduceIndent {indent} {
	global indentationAmount
	return [text::minSpaceForm [string range [text::maxSpaceForm $indent] $indentationAmount end]]
}

proc htmlFirstLineIndent {indent} {
	if {![is::Whitespace [set text [getText [lineStart [getPos]] [getPos]]]]} {return $indent}
	set indent [text::minSpaceForm $indent]
	set text [text::minSpaceForm $text]
	return [string range $indent [string length $text] end]
}

proc htmlIndentChunk {text {tab ""}} {
	upvar $text txt
	if {$tab == ""} {set tab [text::Tab]}
	regsub -all {\[|\]} $txt {\\&} txt
	regsub -all "\r(\[ \t\]*)" $txt {\r[text::minSpaceForm "\1$tab"]} txt
	set txt [subst $txt]
}

#===============================================================================
# Tidy up source
#===============================================================================
proc htmlReformatParagraph {} {htmlTidyUp paragraph}
proc htmlReformatDocument {} {htmlTidyUp document}

proc htmlTidyUp {where} {
	global fillColumn
	htmlTidyUp2 $where $fillColumn
}

proc htmlTidyUp2 {where fillColumn} {
	global HTMLmodeVars htmlElemProc htmlIndentElements indentationAmount
	message "Reformatting"
	set oldfillColumn $fillColumn
	if {$where == "paragraph"} {
		if {[isSelection]} {
			set startPos [getPos]
			set endPos [selEnd]
		} else {
			if {[catch {search -s -f 0 -m 0 -r 1 {^[ \t]*$} [getPos]} sp]} {set sp 0}
			set startPos [nextLineStart [lindex $sp 1]]
			if {[catch {search -s -f 1 -m 0 -r 1 {^[ \t]*$} [getPos]} sp]} {set sp "0 [maxPos]"}
			set endPos [expr [lindex $sp 1] < [maxPos] ? [lindex $sp 1] + 1 : [maxPos]]
		}
		# Avoid doing something inside STYLE and SCRIPT.
		foreach stsc {STYLE SCRIPT} {
			if {[htmlIsInContainer $stsc $startPos]} {
				if {[catch {search -s -f 1 -m 0 -r 0 -i 1 "</$stsc>" $startPos} rrr]} {
					message ""; return
				} else {
					set startPos [lindex $rrr 1]
				}
			}
			if {[htmlIsInContainer $stsc $endPos]} {
				if {[catch {search -s -f 0 -m 0 -r 1 -i 1 "<$stsc\[^<>\]*>" $endPos} rrr]} {
					message ""; return
				} else {
					set endPos [lindex $rrr 0]
				}
			} 
		}
		set ind [htmlFindIndent $startPos]
		set fillColumn [expr $oldfillColumn - [string length [text::maxSpaceForm $ind]]]
		set cr 2
	} else {
		set startPos 0
		set endPos [maxPos]
		set ind ""
		set cr 0
	}
	# Indent region if completely inside STYLE or SCRIPT.
	if {$startPos > $endPos} {::indentRegion; return}
	# Remember position
	set pos [expr [getPos] > $startPos ? [getPos] : $startPos]
	set srem [expr $pos - 20 < $startPos ? $startPos : $pos - 20]
	set remember_str [quote::Regfind [getText $srem $pos ]]
	regsub -all {\?} $remember_str {\\?} remember_str
	regsub -all "\[ \t\r\]+" $remember_str {[ \t\r]+} remember_str
	# To handle indentation
	set indList ""
	foreach i $htmlIndentElements {
		if {$HTMLmodeVars(indent$i)} {lappend indList $i}
	}
	
	# These tags should have a blank line before
	set blBef {TITLE HEAD BODY STYLE H1 H2 H3 H4 H5 H6 P BLOCKQUOTE DIV CENTER PRE MULTICOL OBJECT
	NOEMBED UL OL DIR MENU DL FORM FIELDSET SELECT OPTGROUP TABLE TR FRAMESET NOFRAMES MAP APPLET SCRIPT NOSCRIPT LAYER NOLAYER}
	# These tags should have a cr before
	set crBef {/HTML /HEAD /BODY /STYLE /P /BLOCKQUOTE /DIV ADDRESS /CENTER /PRE /MULTICOL BDO INS DEL HR BASEFONT
	MARQUEE /OBJECT BGSOUND /NOEMBED /UL /OL /DIR /MENU LI /DL DT /FORM /FIELDSET LEGEND /SELECT /OPTGROUP OPTION BUTTON TEXTAREA
	KEYGEN /TABLE /TR CAPTION COL COLGROUP THEAD TBODY TFOOT /FRAMESET FRAME /NOFRAMES /MAP AREA
	/APPLET PARAM /SCRIPT /NOSCRIPT /LAYER ILAYER /NOLAYER BASE ISINDEX LINK META !--}
	# These tags should have a blank line after
	set blAft {/TITLE /HEAD /BODY /STYLE /H1 /H2 /H3 /H4 /H5 /H6 /P /BLOCKQUOTE /DIV /CENTER /PRE /MULTICOL
	/OBJECT /NOEMBED /UL /OL /DIR /MENU /DL /FORM /FIELDSET /SELECT /OPTGROUP /TABLE /TR /FRAMESET /NOFRAMES /MAP
	/APPLET /SCRIPT /NOSCRIPT /LAYER /NOLAYER}
	# These tags should have a cr after
	set crAft {HTML /HTML HEAD BODY STYLE P BLOCKQUOTE DIV /ADDRESS CENTER PRE MULTICOL BR HR WBR BASEFONT
	/MARQUEE OBJECT BGSOUND NOEMBED UL OL DIR MENU /LI DL /DD FORM FIELDSET /LEGEND INPUT SELECT /BUTTON OPTGROUP /TEXTAREA /BDO /INS /DEL KEYGEN
	TABLE TR /CAPTION COL COLGROUP THEAD TBODY TFOOT FRAMESET FRAME NOFRAMES MAP AREA APPLET PARAM
	SCRIPT NOSCRIPT LAYER /ILAYER NOLAYER BASE ISINDEX LINK META !-- !DOCTYPE}
	# Custom elements
	foreach c [array names htmlElemProc] {
		switch [lindex $htmlElemProc($c) 0] {
			htmlBuildCR2Elem {
				lappend blBef $c
				lappend crBef /$c
				lappend blAft /$c
				lappend crAft $c
			}
			htmlBuildCRElem {
				if {[lindex $htmlElemProc($c) 2] == "1"} {
					lappend blBef $c
					lappend blAft /$c
				} else {
					lappend crBef $c
					lappend crAft /$c
				}
			}
			htmlBuildOpening {
				if {[lindex $htmlElemProc($c) 2] == "1"} {lappend crBef $c}
				if {[lindex $htmlElemProc($c) 3] == "1"} {lappend crAft $c}
			}
		}
	}
	set all [concat $blBef $blAft $crBef $crAft]
	set bef [concat $blBef $crBef]
	set aft [concat $blAft $crAft]
	set pos $startPos
	set tmp ""
	set text ""
	while {![catch {search -s -f 1 -m 0 -r 1 {(<!--|<[^<>]+>)} $pos} pos1] && [lindex $pos1 1] <= $endPos} {
		set tag [string toupper [lindex [set wholeTag [string trim [eval getText $pos1] "<>"]] 0]]
		if {$tag != "!--"} {
			set w ""
			set i {0 0}
			# To avoid line breaks inside attributes
			while {[regexp -indices {=\"[^ \"]* [^\"]*\"} $wholeTag i]} {
				append w [string range $wholeTag 0 [expr [lindex $i 0] - 1]]
				regsub -all "\[ \t\r\]+" [string range $wholeTag [lindex $i 0] [lindex $i 1]] "" w1
				append w $w1
				set wholeTag [string range $wholeTag [expr [lindex $i 1] + 1] end]
			}
			set wholeTag $w$wholeTag
		}
		append tmp [getText $pos [lindex $pos1 0]]
		set pos [lindex $pos1 1]			
		if {[lsearch $all $tag] < 0} {
			append tmp <$wholeTag>
			continue
		}
		# cr or blank line before tag
		if {[lsearch $bef $tag] >= 0} {
			regsub -all "\[ \t\]*\r\[ \t\]*" [string trim $tmp] " " tmp
			set tmp [string trimright [breakIntoLines $tmp]]
			regsub -all "" $tmp " " tmp
			regsub -all "\r" $tmp "\r$ind" tmp
			if {![is::Whitespace $tmp]} {set cr 0; append text $ind}
			append text $tmp
			set ble [lsearch $blBef $tag]
			if {$cr == 1 && $ble >= 0 && ([string index $tag 0] != "/" || [lsearch $indList [string range $tag 1 end]] < 0)} {
				append text $ind
			}
			if {$cr == 0} {
				append text \r
				incr cr
				if {$cr == 1 && $ble >= 0} {append text $ind}
			}
			if {$ble >= 0 && $cr < 2} {append text \r; incr cr}
			set tmp <$wholeTag>
			# Take care of comments separately
			if {$tag == "!--"} {
				set tmp "<!--"
				if {[catch {search -s -f 1 -m 0 -r 1 -i 1 -- "-->" $pos} pos2]} {set pos2 "0 $endPos"}
				append text $ind$tmp[getText $pos [set pos [lindex $pos2 1]]]
				set tmp ""
				set cr 0
			}
			# The contents of these tags should be left untouched
			if {[lsearch {SCRIPT STYLE PRE} $tag] >= 0} {
				set tag /$tag
				regsub -all "" $tmp " " tmp
				if {[catch {search -s -f 1 -m 0 -r 1 -i 1 "<$tag>" $pos} pos2]} {set pos2 "0 $endPos"}
				append text $ind$tmp[getText $pos [set pos [lindex $pos2 1]]]
				set tmp ""
				set cr 0
			}
		} else {
			append tmp <$wholeTag>
		}
		# cr or blank line after tag
		if {[lsearch $aft $tag] >= 0} {
			if {[string index $tag 0] == "/" && [lsearch $indList [string range $tag 1 end]] >= 0} {
				set ind [htmlReduceIndent $ind]
				set fillColumn [expr $oldfillColumn - [string length [text::maxSpaceForm $ind]]]
			}
			regsub -all "\[ \t\]*\r\[ \t\]*" [string trim $tmp] " " tmp
			set tmp [string trimright [breakIntoLines $tmp]]
			regsub -all "" $tmp " " tmp
			regsub -all "\r" $tmp "\r$ind" tmp
			if {![is::Whitespace $tmp]} {set cr 0; append text $ind}
			append text $tmp
			set bla [lsearch $blAft $tag]
			if {[lsearch $indList $tag] >= 0} {
				set ind [htmlIncreaseIndent $ind]
				set fillColumn [expr $oldfillColumn - [string length [text::maxSpaceForm $ind]]]
			}
			if {$cr == 0} {
				append text \r
				incr cr
				if {$cr == 1 && $bla >= 0} {append text $ind}
			}
			if {$bla >= 0 && $cr < 2} {append text \r; incr cr}
			set tmp ""
		}
	}
	# Add what's left
	if {$tmp != "" || $pos < $endPos} {
		if {$pos < $endPos} {append tmp [getText $pos $endPos]}
		regsub -all "\[ \t\]*\r\[ \t\]*" [string trim $tmp] " " tmp
		set tmp [string trimright [breakIntoLines $tmp]]
		regsub -all "" $tmp " " tmp
		regsub -all "\r" $tmp "\r$ind" tmp
		if {![is::Whitespace $tmp]} {append text $ind}
		append text $tmp
		if {![is::Whitespace $tmp]} {append text \r}
	}
	replaceText $startPos $endPos $text
	# Go back to previous position.
	if { $remember_str != "" } {
		regexp -indices $remember_str [getText $startPos [set end [getPos]]] wholematch
		set p [expr [info exists wholematch] ? [expr $startPos + 1 + [lindex $wholematch 1]] : $end]
		goto [expr $p >= $end ? $end -1 : $p]
	}
}
